/*****************************************************************************
 *
 * File: Univ7.ddl
 * Version: 7.2.0
 *
 *****************************************************************************
 * Copyright (c) 2010, HART Communication Foundation, All Rights Reserved
 *****************************************************************************
 *
 * Description: This file contains the DDL based model of the HART6 Universal
 *      Command Specification.  There were significant changes in this model
 *      reflecting the increasing focus on the device variable. This resulted
 *      in a hierachal model that matches the one taught in the Writing Device
 *      Description Workshop and the Command Summary Specification.
 *
 * Source Specifications
 *      Command Summary Specification       Rev.9.1
 *      Universal Command Specification     Rev.7.1
 *      Common Tables Specification         Rev.19.0
 *
 *****************************************************************************
 */

/*****************************************************************************
 * The manufacturer line
 *****************************************************************************
 */
MANUFACTURER STANDARD, DEVICE_TYPE _UNIVERSAL, DEVICE_REVISION 7, DD_REVISION 2

/*****************************************************************************
 * Include Files
 *****************************************************************************
 */
#include "macros.h"
#include "methods.h"

/*****************************************************************************
 * Definitions (#define, typedef, enum)
 *****************************************************************************
 */

/*****************************************************************************
 * The following VARIABLEs (response_code, comm_status, and device_status)
 * are common to all HART commands.  These three also have magic significance
 * to DDL masters.  For example, response_code takes on the enumerations
 * defined in the command transaction in progress.
 *
 *****************************************************************************
 * Command Specific Response Codes are encoded in bits #6 - #0 of the
 * first byte of the Response Codes. These codes are 7-bit enumerated
 * values ranging from 0 to 127.  DDL masters must determine the actual
 * command specific enumerations based on the COMMMAND and RESPONSE_CODES
 * currently being communicated.  
 *
 * All COMMAND REPLY declarations must include this VARIABLE as the first
 * parameter.  If in the actual slave communications response the high order
 * bit is set the master must update the value of the comm_status VARIABLE
 * when this happens the value of response_code VARIABLE is not defined.
 * 
 * NOTE - Specific requirements for the definition and use of response codes
 *      are specified in " Command Response Code Specification"
 *****************************************************************************
 */

VARIABLE response_code
{
    LABEL [command_specific_response_code];
    HELP [command_specific_response_code_help];
    CLASS HART & DIAGNOSTIC;
    HANDLING READ;
    TYPE ENUMERATED
    {
        { 0,  [success] },                     /* These are the single defintion errors!!! */
        { 2,  [invalid_selection] },           /* DDL masters must determine the actual  command */
        { 3,  [passed_parameter_too_large] },  /* specific enumerations found in the COMMMAND */
        { 4,  [passed_parameter_too_small] },  /* and RESPONSE_CODES currently being communicated. */
        { 5,  [too_few_data_bytes_recieved] },
        { 6,  [xmtr_specific_command_error] },
        { 7,  [in_write_protect_mode] },
        { 16, [access_restricted] } ,
        { 32, [busy] },
        { 33, [delayed_response_initiated] },
        { 34, [delayed_response_running] },
        { 35, [delayed_response_dead] },
        { 36, [delayed_response_conflict] },
        { 64, [Command_not_implemented] }
    }
}

/*****************************************************************************
 * This VARIABLE never actually is seen in a COMMAND declaration.  DDL masters
 * demultiplex the first byte in all slave responses to update either
 * response_code of comm_status
 * 
 * If in the actual slave communications response the high order
 * bit is set the master must update the value of the comm_status VARIABLE
 * when this happens the value of response_code VARIABLE is not defined.
 *****************************************************************************
 */
VARIABLE comm_status
{
    LABEL [communications_error_summary];
    HELP [communications_error_summary_help];
    CLASS HART & DYNAMIC & DIAGNOSTIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x02, [buffer_overflow],              HARDWARE & CORRECTABLE },
        { 0x08, [longitudinal_parity_error],    HARDWARE & CORRECTABLE },
        { 0x10, [framing_error],                HARDWARE & CORRECTABLE },
        { 0x20, [overrun_error],                HARDWARE & CORRECTABLE },
        { 0x40, [vertical_parity_error],        HARDWARE & CORRECTABLE }
    }
}

/*****************************************************************************
 * request_preambles suggest how many pre-ambles a master should send a field
 * device.  preambles are a physical layer requirement and are used to train
 * the demodulator and the uart and seynchronize to the start of a message
 * 
 * This value is usually 5.  this value must not be abused or utilized to
 * cover up file device software limitations.  large values returned in this
 * VARIABLE reduce communication throughput and must be avoided.
 *
 *****************************************************************************
 */
VARIABLE request_preambles
{
    LABEL [number_of_request_preambles];
    HELP [number_of_request_preambles_help];
    CLASS HART;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER;
}

/*****************************************************************************
 * universal_revision indicates the major revision of the protocol supported
 * by THE field device.  It also allows non-dd capable hosts to select the
 * Universal Commands that the field device supports.  the value retunred by
 * the field device should be 6
 *
 * NOTE - Field Device Developers must support all specifications of a specifc
 * revision of the protocol as indicated by the "Hart Filed Communications 
 * Protocol Specification" document.  A developer is not allowed to  pick 
 * and choose features to support from differnt specification revisions.
 *
 *****************************************************************************
 */
VARIABLE universal_revision
{
    LABEL [universal_revision];
    HELP [universal_revision_help];
    CLASS HART;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER
    {
        MIN_VALUE 7;  // Helps for tokenizer versions earlier than 600, and 
    }                 // hosts that do not support DEFAULT_VALUE
#if __TOKVER__ >= 600
    DEFAULT_VALUE 7;
#endif
}

/*****************************************************************************
 * This variable, along with the Manufacturer ID and Device Type  define a 
 * unique set of commands.  In other words the HART Application Layer included 
 * in this field device (universal common practice, and device specific command
 * and data set. This is a whole number that matches the revision level of the
 * device specification documentation required by the Protocol  the format of
 * this device specific document is found in HCF_LIT-18
 *
 *****************************************************************************
 */
VARIABLE transmitter_revision
{
    LABEL [transmitter_revision];
    HELP [field_device_revision_help];
    CLASS HART;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER;
}

/*****************************************************************************
 * The device/transmitter revision indicates the command set supported by the 
 * this variable is incremented everytime the firmware in the field device 
 * changes (modifications to the firmware may not affect the commands or data
 * accessable via the HART application layer). 
 *
 * This variable could potentially be utilized as a handle for recalling 
 * Field Devices with a software discrepancy."
 *
 *****************************************************************************
 */
VARIABLE software_revision
{
    LABEL [software_revision];
    HELP [field_device_software_revision_help];
    CLASS DEVICE;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER;
}

/*****************************************************************************
 * The hardware revision is utilized to track what ever level of modifications
 * are desired. However, minor changes in components values etc. should not 
 * cause this value to be incremented. 
 *
 * NOTE - please note that in many hardware / software configurations the 
 * software can be removed from the hardware to be upgraded, moved, etc.. In
 * this situation the variable hardware_revision must not reside in the
 * software, but should be readable via from the actual hardware  itself.
 *
 *****************************************************************************
 */
VARIABLE hardware_revision
{
    LABEL [hardware_revision];
    HELP [hardware_revision_help];
    CLASS DEVICE;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER;
}

/*****************************************************************************
 * This variable should be written into the Field Device in the manufacturing 
 * process. Every device shipped must have a different value for this variable
 * This variable alonf with the manufacturer_id and device_type. indicate one 
 * and only one unique device.  many host systems combine these three 
 * variables to track the devices life cycle.
 *
 *****************************************************************************
 */
VARIABLE device_id
{
    LABEL [device_identification];
    HELP [field_device_identification_help];
    CLASS DEVICE;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER (3);
}

/*****************************************************************************
 * response preambles are set by command 59 and indicate the number of 
 * preambles the master needs for is physical layer.  preambles are a physical
 * layer requirement and are used to train the demodulator and the uart 
 * and seynchronize to the start of a message
 * 
 * This value is usually 5.  this value must not vary from message to message,
 * it must not be abused or utilized to cover up file device software 
 * limitations.  large values returned in this VARIABLE reduce communication
 * throughput and must be avoided.
 *
 *****************************************************************************
 */
VARIABLE response_preambles
{
    LABEL [number_of_response_preambles];
    HELP [number_of_response_preambles_help];
    CLASS HART;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER
    {
        MIN_VALUE  5;
        MAX_VALUE  20;
    }
}

/*****************************************************************************
 * This variable indicates the last Device Variable code that a host application
 * should expect to be found in the field device (other than the Standard
 * Device Variables like Battery Life, Percent of Range, Loop current, 
 * PV, SV, TV, and QV).  
 *****************************************************************************
 */
VARIABLE max_num_device_variables
{
    LABEL [maximum_number_of_device_variables];
    HELP [maximum_number_of_device_variables_help];
    CLASS DEVICE;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER;
}

/*****************************************************************************
 * This variable tracks the number of times a device's configuration has 
 * changed. Its value increments each time a configuration change is made.
 *****************************************************************************
 */
VARIABLE config_change_counter
{
    LABEL [configuration_change_counter];
    HELP [configuration_change_counter_help];
    CLASS DEVICE & DYNAMIC;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER (2);
}

/**********************************************************************
 * Notes on identity commands
 *
 * The VARIABLE manufacturer_id, combined with the VARIABLE
 * device_type, will uniquely identify a type of Field Device. The
 * combination of these two VARIABLEs will have only one unique 
 * Device Description, though there may be revisions of the Device
 * Description.
 *
 * The VARIABLE manufacturer_id, combined with the VARIABLE
 * device_type, combined with the VARIABLE device_revision,
 * will identify the exact revision of the Device Description for
 * this type of Field Device.
 */
COMMAND read_unique_identifier 
{
    NUMBER 0;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            IDENTIFY_REPLY_7
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
    }
}

VARIABLE tag
{
    LABEL [tag];
    HELP [tag_help];
    CLASS DEVICE;
    TYPE PACKED_ASCII (8);
}

COMMAND read_unique_identifier_with_tag
{
    NUMBER 11;
    OPERATION COMMAND;
    TRANSACTION
    {
        REQUEST
        {
            tag
        }
        REPLY
        {
            IDENTIFY_REPLY_7
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS, [no_command_specific_errors];
    }
}

VARIABLE longTag
{
    LABEL [long_tag];
    HELP [long_tag_help];
    CLASS DEVICE;
    TYPE ASCII (32);
}

COMMAND read_unique_identifier_with_long_tag
{
    NUMBER 21;
    OPERATION COMMAND;
    TRANSACTION
    {
        REQUEST
        {
            longTag
        }
        REPLY
        {
            IDENTIFY_REPLY_7
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS, [no_command_specific_errors];
    }
}

/**********************************************************************
 * Process data commands
 */

COMMAND read_pv 
{
    NUMBER 1;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, 
            PV.DEVICE_VARIABLE.DIGITAL_UNITS, PV.DEVICE_VARIABLE.DIGITAL_VALUE
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,      [no_command_specific_errors];
        6,  MISC_ERROR,   [xmtr_specific_command_error];
        8,  MISC_WARNING, [update_failure];
        16, MODE_ERROR,   [access_restricted];
    }
}

COMMAND read_pv_current_and_percent_range 
{
    NUMBER 2;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY 
        {
            response_code, device_status, 
            PV.DAQ.ANALOG_VALUE, 
            PV.RANGING.PERCENT_RANGE
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,      [no_command_specific_errors];
        6,  MISC_ERROR,   [xmtr_specific_command_error];
        8,  MISC_WARNING, [update_failure];
        16, MODE_ERROR,   [access_restricted];
    }
}

COMMAND read_dynamic_variables_and_pv_current 
{
    NUMBER 3;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status,
            PV.DAQ.ANALOG_VALUE,
            PV.DEVICE_VARIABLE.DIGITAL_UNITS, PV.DEVICE_VARIABLE.DIGITAL_VALUE,
            SV.DEVICE_VARIABLE.DIGITAL_UNITS, SV.DEVICE_VARIABLE.DIGITAL_VALUE,
            TV.DEVICE_VARIABLE.DIGITAL_UNITS, TV.DEVICE_VARIABLE.DIGITAL_VALUE,
            QV.DEVICE_VARIABLE.DIGITAL_UNITS, QV.DEVICE_VARIABLE.DIGITAL_VALUE
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,      [no_command_specific_errors];
        6,  MISC_ERROR,   [xmtr_specific_command_error];
        8,  MISC_WARNING, [update_failure];
        16, MODE_ERROR,   [access_restricted];
    }
}


VARIABLE device_variable_code_1
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_2
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_3
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_4
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_5
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_6
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_7
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

VARIABLE device_variable_code_8
{
    CLASS LOCAL;
    TYPE INDEX deviceVariables;
}

//#if __TOKVER__ >= 800
VARIABLE time_stamp 
{    
    LABEL "Time Stamp";
    HELP "Relative time value on network";
    CLASS DEVICE & DYNAMIC;
    HANDLING READ;
    TYPE TIME_VALUE;
}
//#endif
    
COMMAND read_device_variables_and_status
{
    NUMBER 9;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST
        {
            device_variable_code_1 (INFO,INDEX),
            device_variable_code_2 (INFO,INDEX),
            device_variable_code_3 (INFO,INDEX),
            device_variable_code_4 (INFO,INDEX)
#if __TOKVER__ >= 800
            device_variable_code_5 (INFO,INDEX),
            device_variable_code_6 (INFO,INDEX),
            device_variable_code_7 (INFO,INDEX), 
            device_variable_code_8 (INFO,INDEX)  
#endif
        }
        REPLY
        {
            response_code, device_status,
            extended_fld_device_status,

            device_variable_code_1 (INFO,INDEX),
            deviceVariables[device_variable_code_1].CLASSIFICATION,
            deviceVariables[device_variable_code_1].DIGITAL_UNITS,
            deviceVariables[device_variable_code_1].DIGITAL_VALUE,
            deviceVariables[device_variable_code_1].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_1].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_1].DEVICE_FAMILY_STATUS <0x0F>,

            device_variable_code_2 (INFO,INDEX),
            deviceVariables[device_variable_code_2].CLASSIFICATION,
            deviceVariables[device_variable_code_2].DIGITAL_UNITS,
            deviceVariables[device_variable_code_2].DIGITAL_VALUE,
            deviceVariables[device_variable_code_2].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_2].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_2].DEVICE_FAMILY_STATUS <0x0F>,

            device_variable_code_3 (INFO,INDEX),
            deviceVariables[device_variable_code_3].CLASSIFICATION,
            deviceVariables[device_variable_code_3].DIGITAL_UNITS,
            deviceVariables[device_variable_code_3].DIGITAL_VALUE,
            deviceVariables[device_variable_code_3].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_3].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_3].DEVICE_FAMILY_STATUS <0x0F>,

            device_variable_code_4 (INFO,INDEX),
            deviceVariables[device_variable_code_4].CLASSIFICATION,
            deviceVariables[device_variable_code_4].DIGITAL_UNITS,
            deviceVariables[device_variable_code_4].DIGITAL_VALUE,
            deviceVariables[device_variable_code_4].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_4].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_4].DEVICE_FAMILY_STATUS <0x0F>
#if __TOKVER__ >= 800
            device_variable_code_5 (INFO,INDEX),
            deviceVariables[device_variable_code_5].CLASSIFICATION,
            deviceVariables[device_variable_code_5].DIGITAL_UNITS,
            deviceVariables[device_variable_code_5].DIGITAL_VALUE,
            deviceVariables[device_variable_code_5].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_5].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_5].DEVICE_FAMILY_STATUS <0x0F>

            device_variable_code_6 (INFO,INDEX),
            deviceVariables[device_variable_code_6].CLASSIFICATION,
            deviceVariables[device_variable_code_6].DIGITAL_UNITS,
            deviceVariables[device_variable_code_6].DIGITAL_VALUE,
            deviceVariables[device_variable_code_6].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_6].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_6].DEVICE_FAMILY_STATUS <0x0F>,

            device_variable_code_7 (INFO,INDEX),
            deviceVariables[device_variable_code_7].CLASSIFICATION,
            deviceVariables[device_variable_code_7].DIGITAL_UNITS,
            deviceVariables[device_variable_code_7].DIGITAL_VALUE,
            deviceVariables[device_variable_code_7].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_7].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_7].DEVICE_FAMILY_STATUS <0x0F>,

            device_variable_code_8 (INFO,INDEX),
            deviceVariables[device_variable_code_8].CLASSIFICATION,
            deviceVariables[device_variable_code_8].DIGITAL_UNITS,
            deviceVariables[device_variable_code_8].DIGITAL_VALUE,
            deviceVariables[device_variable_code_8].DATA_QUALITY <0xC0>,
            deviceVariables[device_variable_code_8].LIMIT_STATUS <0x30>,
            deviceVariables[device_variable_code_8].DEVICE_FAMILY_STATUS <0x0F>,
            time_stamp
#endif   
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,          [no_command_specific_errors];
        2,  DATA_ENTRY_ERROR, [invalid_selection];
        5,  DATA_ENTRY_ERROR, [too_few_data_bytes_recieved];
        6,  MISC_ERROR,       [xmtr_specific_command_error];
        8,  MISC_WARNING,     [update_failure];
        14, MISC_WARNING,     "Dynamic Variables Returned for Device Variables"; // CPK - add this to standard dictionary
        16, MODE_ERROR,       [access_restricted];
        30, MISC_WARNING,     "Command Response Truncated"; // CPK  - add this to standard dictionary
    }
}
      
/**********************************************************************
 * Polling address and loop current mode commands
 */

VARIABLE polling_address
{
    LABEL [polling_address];
    HELP [polling_address_help];
    CLASS HART;
    TYPE UNSIGNED_INTEGER
    {
        MAX_VALUE 63;
    }
}

COMMAND write_polling_address
{
    NUMBER 6;
    OPERATION WRITE;
    TRANSACTION
    {
        REQUEST
        {
            polling_address, loop_current_mode
        }
        REPLY
        {
            response_code, device_status, polling_address, loop_current_mode
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,          [no_command_specific_errors];
        2,  DATA_ENTRY_ERROR, [invalid_poll_address_selection];
        5,  DATA_ENTRY_ERROR, [too_few_data_bytes_recieved];
        6,  MISC_ERROR,       [xmtr_specific_command_error];
        7,  MODE_ERROR,       [in_write_protect_mode];
        12, MODE_ERROR,       [invalid_mode_selection];
        16, MODE_ERROR,       [access_restricted];
        32, MODE_ERROR,       [busy];
    }
}

COMMAND read_loop_configuration
{
    NUMBER 7;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, polling_address, loop_current_mode
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
    }
}

COMMAND read_dynamic_variable_classification
{
    NUMBER 8;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status,
            PV.DEVICE_VARIABLE.CLASSIFICATION,
            SV.DEVICE_VARIABLE.CLASSIFICATION,
            TV.DEVICE_VARIABLE.CLASSIFICATION,
            QV.DEVICE_VARIABLE.CLASSIFICATION
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
    }
}

COMMAND read_pv_sensor_info
{
    NUMBER 14;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status,
            PV.DEVICE_VARIABLE.SENSOR_SERIAL_NUMBER, 
            PV.DEVICE_VARIABLE.DIGITAL_UNITS,
            PV.DEVICE_VARIABLE.UPPER_SENSOR_LIMIT,
            PV.DEVICE_VARIABLE.LOWER_SENSOR_LIMIT,
            PV.DEVICE_VARIABLE.MINIMUM_SPAN
        }
    }
    RESPONSE_CODES
    {
        0, SUCCESS,     [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
        32, MODE_ERROR, [busy];
    }
}

COMMAND read_pv_output_info
{
    NUMBER 15;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status,
            PV.DAQ.ALARM_CODE, 
            PV.RANGING.TRANSFER_FUNCTION,
            PV.RANGING.RANGE_UNITS, 
            PV.RANGING.UPPER_RANGE_VALUE,
            PV.RANGING.LOWER_RANGE_VALUE,
            PV.DEVICE_VARIABLE.DAMPING_VALUE,
            write_protect, 
            250,
            PV.DAQ.ANALOG_CHANNEL_FLAGS
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
        32, MODE_ERROR, [busy];
    }
}

VARIABLE message
{
    LABEL [message];
    HELP [message_help];
    CLASS DEVICE;
    TYPE PACKED_ASCII (32);
}

COMMAND read_message
{
    NUMBER 12;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, message
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
        32, MODE_ERROR, [busy];
    }
}

COMMAND write_message
{
    NUMBER 17;
    OPERATION WRITE;
    TRANSACTION
    {
        REQUEST
        {
            message
        }
        REPLY
        {
            response_code, device_status, message
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,          [no_command_specific_errors];
        5,  DATA_ENTRY_ERROR, [too_few_data_bytes_recieved];
        6,  MISC_ERROR,       [xmtr_specific_command_error];
        7,  MODE_ERROR,       [in_write_protect_mode];
        16, MODE_ERROR,       [access_restricted];
        32, MODE_ERROR,       [busy];
     }
}

VARIABLE descriptor
{
    LABEL [descriptor];
    HELP [discriptor_help];
    CLASS DEVICE;
    TYPE PACKED_ASCII (16);
}

VARIABLE date
{
    LABEL [date];
    HELP [date_help];
    CLASS DEVICE;
    TYPE DATE;
}

COMMAND read_tag_descriptor_date
{
    NUMBER 13;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, tag, descriptor, date
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
        32, MODE_ERROR, [busy];
    }
}

COMMAND write_tag_descriptor_date
{
    NUMBER 18;
    OPERATION WRITE;
    TRANSACTION
    {
        REQUEST
        {
            tag, descriptor, date
        }
        REPLY
        {
            response_code, device_status, tag, descriptor, date
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,          [no_command_specific_errors];
        5,  DATA_ENTRY_ERROR, [too_few_data_bytes_recieved];
        6,  MISC_ERROR,       [xmtr_specific_command_error];
        7,  MODE_ERROR,       [in_write_protect_mode];
        9,  DATA_ENTRY_ERROR, [invalid_date_code_detected];
        16, MODE_ERROR,       [access_restricted];
        32, MODE_ERROR,       [busy];
    }
}

VARIABLE final_assembly_number
{
    LABEL [final_assembly_number];
    HELP [final_assembly_number_help];
    CLASS DEVICE;
    TYPE UNSIGNED_INTEGER (3)
    {
        MAX_VALUE 16777215;
        EDIT_FORMAT "8u" ;
    }
}

COMMAND read_final_assembly_number
{
    NUMBER 16;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, final_assembly_number
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
        32, MODE_ERROR, [busy];
    }
}

COMMAND write_final_assembly_number
{
    NUMBER 19;
    OPERATION WRITE;
    TRANSACTION
    {
        REQUEST
        {
            final_assembly_number
        }
        REPLY
        {
            response_code, device_status, final_assembly_number
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,          [no_command_specific_errors];
        5,  DATA_ENTRY_ERROR, [too_few_data_bytes_recieved];
        6,  MISC_ERROR,       [xmtr_specific_command_error];
        7,  MODE_ERROR,       [in_write_protect_mode];
        16, MODE_ERROR,       [access_restricted];
        32, MODE_ERROR,       [busy];
    }
}

COMMAND read_long_tag
{
    NUMBER 20;
    OPERATION READ;
    TRANSACTION
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, longTag
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,    [no_command_specific_errors];
        16, MODE_ERROR, [access_restricted];
        32, MODE_ERROR, [busy];
    }
}

COMMAND write_long_tag
{
    NUMBER 22;
    OPERATION WRITE;
    TRANSACTION
    {
        REQUEST
        {
            longTag
        }
        REPLY
        {
            response_code, device_status, longTag
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,          [no_command_specific_errors];
        5,  DATA_ENTRY_ERROR, [too_few_data_bytes_recieved];
        6,  MISC_ERROR,       [xmtr_specific_command_error];
        7,  MODE_ERROR,       [in_write_protect_mode];
        16, MODE_ERROR,       [access_restricted];
        32, MODE_ERROR,       [busy];
        33, MODE_ERROR,       [delayed_response_initiated];
        34, MODE_ERROR,       [delayed_response_running];
        35, MODE_ERROR,       [delayed_response_dead];
        36, MODE_ERROR,       [delayed_response_conflict];
    }
}

COMMAND reset_configuration_change_flag
{
    NUMBER 38;
    OPERATION COMMAND;
    TRANSACTION 0
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, config_change_counter
        }
    }
    TRANSACTION 1
    {
        REQUEST
        {
            config_change_counter (INFO)
        }
        REPLY
        {
            response_code, device_status, config_change_counter (INFO)
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,            [no_command_specific_errors];
        5,  DATA_ENTRY_ERROR,   [too_few_data_bytes_recieved];
        6,  MISC_ERROR,         [xmtr_specific_command_error];
        7,  MODE_ERROR,         [in_write_protect_mode];
        9,  MISC_ERROR,         "Configuration Change Counter Mismatch";
        16, MODE_ERROR,         [access_restricted];
    }
}

VARIABLE device_specific_status_0
{
    LABEL [device_specific_status_0];
    HELP [field_device_status_0_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat0_1], MISC }, { 0x02, [fld_dev_stat0_2], MISC }, { 0x04, [fld_dev_stat0_3], MISC },
        { 0x08, [fld_dev_stat0_4], MISC }, { 0x10, [fld_dev_stat0_5], MISC }, { 0x20, [fld_dev_stat0_6], MISC },
        { 0x40, [fld_dev_stat0_7], MISC }, { 0x80, [fld_dev_stat0_8], MISC }
    }
}

VARIABLE device_specific_status_1
{
    LABEL [device_specific_status_1];
    HELP [field_device_status_1_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat1_1], MISC }, { 0x02, [fld_dev_stat1_2], MISC }, { 0x04, [fld_dev_stat1_3], MISC },
        { 0x08, [fld_dev_stat1_4], MISC }, { 0x10, [fld_dev_stat1_5], MISC }, { 0x20, [fld_dev_stat1_6], MISC },
        { 0x40, [fld_dev_stat1_7], MISC }, { 0x80, [fld_dev_stat1_8], MISC }
    }
}

VARIABLE device_specific_status_2
{
    LABEL [device_specific_status_2];
    HELP [field_device_status_2_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat2_1], MISC }, { 0x02, [fld_dev_stat2_2], MISC }, { 0x04, [fld_dev_stat2_3], MISC },
        { 0x08, [fld_dev_stat2_4], MISC }, { 0x10, [fld_dev_stat2_5], MISC }, { 0x20, [fld_dev_stat2_6], MISC },
        { 0x40, [fld_dev_stat2_7], MISC }, { 0x80, [fld_dev_stat2_8], MISC }
    }
}

VARIABLE device_specific_status_3
{
    LABEL [device_specific_status_3];
    HELP [field_device_status_3_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat3_1], MISC }, { 0x02, [fld_dev_stat3_2], MISC }, { 0x04, [fld_dev_stat3_3], MISC },
        { 0x08, [fld_dev_stat3_4], MISC }, { 0x10, [fld_dev_stat3_5], MISC }, { 0x20, [fld_dev_stat3_6], MISC },
        { 0x40, [fld_dev_stat3_7], MISC }, { 0x80, [fld_dev_stat3_8], MISC }
    }
}

VARIABLE device_specific_status_4
{
    LABEL [device_specific_status_4];
    HELP [field_device_status_4_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat4_1], MISC }, { 0x02, [fld_dev_stat4_2], MISC }, { 0x04, [fld_dev_stat4_3], MISC },
        { 0x08, [fld_dev_stat4_4], MISC }, { 0x10, [fld_dev_stat4_5], MISC }, { 0x20, [fld_dev_stat4_6], MISC },
        { 0x40, [fld_dev_stat4_7], MISC }, { 0x80, [fld_dev_stat4_8], MISC }
    }
}

VARIABLE device_specific_status_5
{
    LABEL [device_specific_status_5];
    HELP [field_device_status_5_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat5_1], MISC }, { 0x02, [fld_dev_stat5_2], MISC }, { 0x04, [fld_dev_stat5_3], MISC },
        { 0x08, [fld_dev_stat5_4], MISC }, { 0x10, [fld_dev_stat5_5], MISC }, { 0x20, [fld_dev_stat5_6], MISC },
        { 0x40, [fld_dev_stat5_7], MISC }, { 0x80, [fld_dev_stat5_8], MISC }
    }
}

VARIABLE device_specific_status_14
{
    LABEL [device_specific_status_14];
    HELP [field_device_status_14_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat14_1], MISC }, { 0x02, [fld_dev_stat14_2], MISC }, { 0x04, [fld_dev_stat14_3], MISC },
        { 0x08, [fld_dev_stat14_4], MISC }, { 0x10, [fld_dev_stat14_5], MISC }, { 0x20, [fld_dev_stat14_7], MISC },
        { 0x40, [fld_dev_stat14_7], MISC }, { 0x80, [fld_dev_stat14_8], MISC }
    }
}

VARIABLE device_specific_status_15
{
    LABEL [device_specific_status_15];
    HELP [field_device_status_15_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat15_1], MISC }, { 0x02, [fld_dev_stat15_2], MISC }, { 0x04, [fld_dev_stat15_3], MISC },
        { 0x08, [fld_dev_stat15_4], MISC }, { 0x10, [fld_dev_stat15_5], MISC }, { 0x20, [fld_dev_stat15_6], MISC },
        { 0x40, [fld_dev_stat15_7], MISC }, { 0x80, [fld_dev_stat15_8], MISC }
    }
}

VARIABLE device_specific_status_16
{
    LABEL [device_specific_status_16];
    HELP [field_device_status_16_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat16_1], MISC }, { 0x02, [fld_dev_stat16_2], MISC }, { 0x04, [fld_dev_stat16_3], MISC },
        { 0x08, [fld_dev_stat16_4], MISC }, { 0x10, [fld_dev_stat16_5], MISC }, { 0x20, [fld_dev_stat16_6], MISC },
        { 0x40, [fld_dev_stat16_7], MISC }, { 0x80, [fld_dev_stat16_8], MISC }
    }
}

VARIABLE device_specific_status_17
{
    LABEL [device_specific_status_17];
    HELP [field_device_status_17_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat17_1], MISC }, { 0x02, [fld_dev_stat17_2], MISC }, { 0x04, [fld_dev_stat17_3], MISC },
        { 0x08, [fld_dev_stat17_4], MISC }, { 0x10, [fld_dev_stat17_5], MISC }, { 0x20, [fld_dev_stat17_6], MISC },
        { 0x40, [fld_dev_stat17_7], MISC }, { 0x80, [fld_dev_stat17_8], MISC }
    }
}

VARIABLE device_specific_status_18
{
    LABEL [device_specific_status_18];
    HELP [field_device_status_18_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat18_1], MISC }, { 0x02, [fld_dev_stat18_2], MISC }, { 0x04, [fld_dev_stat18_3], MISC },
        { 0x08, [fld_dev_stat18_4], MISC }, { 0x10, [fld_dev_stat18_5], MISC }, { 0x20, [fld_dev_stat18_6], MISC },
        { 0x40, [fld_dev_stat18_7], MISC }, { 0x80, [fld_dev_stat18_8], MISC }
    }
}

VARIABLE device_specific_status_19
{
    LABEL [device_specific_status_19];
    HELP [field_device_status_19_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat19_1], MISC }, { 0x02, [fld_dev_stat19_2], MISC }, { 0x04, [fld_dev_stat19_3], MISC },
        { 0x08, [fld_dev_stat19_4], MISC }, { 0x10, [fld_dev_stat19_5], MISC }, { 0x20, [fld_dev_stat19_6], MISC },
        { 0x40, [fld_dev_stat19_7], MISC }, { 0x80, [fld_dev_stat19_8], MISC }
    }
}

VARIABLE device_specific_status_20
{
    LABEL [device_specific_status_20];
    HELP [field_device_status_20_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat20_1], MISC }, { 0x02, [fld_dev_stat20_2], MISC }, { 0x04, [fld_dev_stat20_3], MISC },
        { 0x08, [fld_dev_stat20_4], MISC }, { 0x10, [fld_dev_stat20_5], MISC }, { 0x20, [fld_dev_stat20_6], MISC },
        { 0x40, [fld_dev_stat20_7], MISC }, { 0x80, [fld_dev_stat20_8], MISC }
    }
}

VARIABLE device_specific_status_21
{
    LABEL [device_specific_status_21];
    HELP [field_device_status_21_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat21_1], MISC }, { 0x02, [fld_dev_stat21_2], MISC }, { 0x04, [fld_dev_stat21_3], MISC },
        { 0x08, [fld_dev_stat21_4], MISC }, { 0x10, [fld_dev_stat21_5], MISC }, { 0x20, [fld_dev_stat21_6], MISC },
        { 0x40, [fld_dev_stat21_7], MISC }, { 0x80, [fld_dev_stat21_8], MISC }
    }
}

VARIABLE device_specific_status_22
{
    LABEL [device_specific_status_22];
    HELP [field_device_status_22_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat22_1], MISC }, { 0x02, [fld_dev_stat22_2], MISC }, { 0x04, [fld_dev_stat22_3], MISC },
        { 0x08, [fld_dev_stat22_4], MISC }, { 0x10, [fld_dev_stat22_5], MISC }, { 0x20, [fld_dev_stat22_6], MISC },
        { 0x40, [fld_dev_stat22_7], MISC }, { 0x80, [fld_dev_stat22_8], MISC }
    }
}

VARIABLE device_specific_status_23
{
    LABEL [device_specific_status_23];
    HELP [field_device_status_23_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat23_1], MISC }, { 0x02, [fld_dev_stat23_2], MISC }, { 0x04, [fld_dev_stat23_3], MISC },
        { 0x08, [fld_dev_stat23_4], MISC }, { 0x10, [fld_dev_stat23_5], MISC }, { 0x20, [fld_dev_stat23_6], MISC },
        { 0x40, [fld_dev_stat23_7], MISC }, { 0x80, [fld_dev_stat23_8], MISC }
    }
}

VARIABLE device_specific_status_24
{
    LABEL [device_specific_status_24];
    HELP [field_device_status_24_help];
    CLASS DEVICE & DIAGNOSTIC & DYNAMIC;
    HANDLING READ;
    TYPE BIT_ENUMERATED
    {
        { 0x01, [fld_dev_stat24_1], MISC }, { 0x02, [fld_dev_stat24_2], MISC }, { 0x04, [fld_dev_stat24_3], MISC },
        { 0x08, [fld_dev_stat24_4], MISC }, { 0x10, [fld_dev_stat24_5], MISC }, { 0x20, [fld_dev_stat24_6], MISC },
        { 0x40, [fld_dev_stat24_7], MISC }, { 0x80, [fld_dev_stat24_8], MISC }
    }
}

/*****************************************************************************
 * Command 48
 *
 * This is a generic definition of command 48 and most implementations
 * truncate the command when IMPORTING
 * 
 * NOTE 1 - the VARIABLE's device_specific_statusXX are normally REDEFINE'd by the
 * device DD 
 *
 * NOTE 2 - All devices must support at least bytes 0-8 including extended_fld_device_status,
 * operatingMode (which must be 0x00) and standardized_status_0.  
 */ 
COMMAND read_additional_device_status
{
    NUMBER 48;
    OPERATION READ;
    TRANSACTION 0
    {
        REQUEST {}
        REPLY
        {
            response_code, device_status, 
            device_specific_status_0, device_specific_status_1, device_specific_status_2,
            device_specific_status_3, device_specific_status_4, device_specific_status_5,

            extended_fld_device_status, 0x00,

            standardized_status_0, standardized_status_1, analog_channel_saturated1,
            standardized_status_2, standardized_status_3, analog_channel_fixed1,

            device_specific_status_14, device_specific_status_15, device_specific_status_16,
            device_specific_status_17, device_specific_status_18, device_specific_status_19,
            device_specific_status_20, device_specific_status_21, device_specific_status_22,
            device_specific_status_23, device_specific_status_24
        }
    }
    TRANSACTION 1
    {
        REQUEST
        {
            device_specific_status_0 (INFO), device_specific_status_1 (INFO), device_specific_status_2 (INFO),
            device_specific_status_3 (INFO), device_specific_status_4 (INFO), device_specific_status_5 (INFO),

            extended_fld_device_status (INFO), 0x00,

            standardized_status_0 (INFO), standardized_status_1 (INFO), analog_channel_saturated1 (INFO),
            standardized_status_2 (INFO), standardized_status_3 (INFO), analog_channel_fixed1 (INFO),

            device_specific_status_14(INFO), device_specific_status_15(INFO), device_specific_status_16(INFO),
            device_specific_status_17(INFO), device_specific_status_18(INFO), device_specific_status_19(INFO),
            device_specific_status_20(INFO), device_specific_status_21(INFO), device_specific_status_22(INFO),
            device_specific_status_23(INFO), device_specific_status_24(INFO)
        }
        REPLY
        {
            response_code, device_status, 
            device_specific_status_0 (INFO), device_specific_status_1 (INFO), device_specific_status_2 (INFO),
            device_specific_status_3 (INFO), device_specific_status_4 (INFO), device_specific_status_5 (INFO),

            extended_fld_device_status (INFO), 0x00,

            standardized_status_0 (INFO), standardized_status_1 (INFO), analog_channel_saturated1 (INFO),
            standardized_status_2 (INFO), standardized_status_3 (INFO), analog_channel_fixed1 (INFO),

            device_specific_status_14 (INFO), device_specific_status_15 (INFO), device_specific_status_16 (INFO),
            device_specific_status_17 (INFO), device_specific_status_18 (INFO), device_specific_status_19 (INFO),
            device_specific_status_20 (INFO), device_specific_status_21 (INFO), device_specific_status_22 (INFO),
            device_specific_status_23 (INFO), device_specific_status_24 (INFO)
        }
    }
    RESPONSE_CODES
    {
        0,  SUCCESS,            [no_command_specific_errors];
        5,  DATA_ENTRY_ERROR,   [too_few_data_bytes_recieved];   
        6,  MISC_ERROR,         [xmtr_specific_command_error];
        8,  MISC_WARNING,       [update_in_progress];
        14, MISC_WARNING,       "Status bytes mismatch";
        16, MODE_ERROR,         [access_restricted];
    }
}